# -*- coding: utf-8 -*-
"""
utils.py
補助関数
"""

import os, sys
import math
import numpy as np

Y0 = 10000

# データ入力
def read_data(path, a):
    if not os.path.exists(path):
        print('file %s not found.' % path)
        sys.exit()

    with open(path, 'rt', encoding='utf-8') as fp:
        # 第1行
        header = fp.readline().split()  # データ数, 開始年(西暦)
        assert (len(header) > 1)
        n0 = int(header[1]) + Y0      # 開始年(補正後、紀元前も正の値にするため)
        n1 = n0 + int(header[0]) - 1  # 終了年(補正後)
        assert (n0 >= 0) and (n0 < len(a))
        assert (n1 >= 0) and (n1 < len(a))

        # 本体
        lines = fp.readlines()
        i = n0
        for line in lines:
            linedata = line.split()
            if (len(linedata) == 5):
                assert(linedata[0] == linedata[4])  # 第1データと第5データは同じ
            a[i] = float(linedata[0]) * 0.01  # mmに変換
            i += 1
        assert (i == n1 + 1)  # データ数チェック

    return n0, n1

# 移動平均(moving average)
# 入力: a, 出力: b
def ma(a, b, n0, n1, lma, div):
    assert (n0 >= 0) and (n1 >= 0) and (n1 >= n0)
    assert (lma % 2 == 1)

    # 作業配列
    x = a.copy()

    if lma > 1:
        mma = (lma - 1) // 2

        # 単純移動平均
        w = np.ones(lma) / lma
        x[n0: n1 + 1] = np.convolve(a[n0: n1 + 1], w, mode='same')

        # 単純移動平均（両端以外）
        #for i in range(n0 + mma, n1 - mma + 1):
        #    x[i] = np.sum(a[i - mma: i + mma + 1]) / lma

        # 両端: 内側の値をコピー（両端の減衰を避けるため）
        for i in range(n0, n0 + mma):
            x[i] = x[n0 + mma]
        for i in range(n1 - mma + 1, n1 + 1):
            x[i] = x[n1 - mma]

    if div:
        b[n0: n1 + 1] = a[n0: n1 + 1] / x[n0: n1 + 1]  # 移動平均で割る
    else:
        b[n0: n1 + 1] = x[n0: n1 + 1]                  # 移動平均をコピー

# データ間平均, データが1個以上存在する年のみ集計する
def average(ndata, f, fmean, n0data, n1data):
    num = np.zeros(len(fmean), dtype=np.int32)

    # 総和
    for idata in range(ndata):
        n0 = n0data[idata]
        n1 = n1data[idata]
        fmean[n0: n1 + 1] += f[idata][n0: n1 + 1]
        num[n0: n1 + 1] += 1

    # 平均
    for i in range(len(fmean)):
        fmean[i] = (fmean[i] / num[i]) if (num[i] > 0) else 0

# t値, scalar
def tf(r, n):
    t = 0
    if (n > 1) and (r > 0) and (r < 1 - 1e-6):
        t = r * math.sqrt((n - 2) / (1 - r * r))

    return t

# 照合
# 出力: リスト配列 [変動量, 重複年数, r, t]
def compare(b1, b2, n0_1, n1_1, n0_2, n1_2, max_offset, min_overlap):
    cmp = []
    for offset in range(-max_offset, +max_offset + 1):
        n0 = max(n0_1, n0_2 + offset)
        n1 = min(n1_1, n1_2 + offset)
        if n1 - n0 + 1 >= min_overlap:
            r = np.corrcoef(b1[n0: n1 + 1], b2[n0 - offset: n1 + 1 - offset])[0, 1]
            t = tf(r, n1 - n0 + 1)
            cmp.append([offset, n1 - n0 + 1, r, t])
    if (len(cmp) == 0):
        print('>>> compare: no data.')

    return cmp
